<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use Illuminate\Support\Facades\Auth;
use App\User;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Tymon\JWTAuth\Facades\JWTAuth;

class AuthController extends Controller

{
    /**
     * Create a new AuthController instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth:api', ['except' => ['login', 'register']]);
    }

    /**
     * Get a JWT via given credentials.
     *
     * @return \Illuminate\Http\JsonResponse
     */

    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required',
            'password' => 'required|string|min:6',
        ]);

        $credentials = ['cCredUsuario' => $request->email, 'password' => $request->password];

        if ($validator->fails()) {
            return response()->json($validator->errors(), 422);
        }

        $value = $this->verificarAcceso($request->email, $request->modulo);

        if ($value == 0) {
            return response()->json(['error' => 'El usuario no existe en nuestros registros.'], 403);
        }

        if (!$token = auth()->attempt($credentials)) {
            return response()->json(['error' => 'Verifica tu usuario y contraseña'], 401);
        }
        return $this->createNewToken($token,$request->modulo);
    }

    public function verificarAcceso($cCredUsuario, $sistema)
    {
        $credencial = DB::select('call _CREDENCIALES(?,?,?)', array($cCredUsuario, $sistema,'ConsultarxcredUsuario'));

        $access = count($credencial);

        return $access;
    }

    public function verificarLinkAcceso($cCredUsuario, $sistema)
    {
        $credencial = DB::select('call _CREDENCIALES(?,?,?)', array($cCredUsuario, $sistema,'ConsultarxcredUsuario'));


        return $credencial;
    }

    /**
     * Register a User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|between:2,100',
            'email' => 'required|string|email|max:100|unique:users',
            'password' => 'required|string|confirmed|min:6',
        ]);

        if ($validator->fails()) {
            return response()->json($validator->errors()->toJson(), 400);
        }

        $user = User::create(array_merge(
            $validator->validated(),
            ['password' => bcrypt($request->password)]
        ));

        return response()->json([
            'message' => 'El usuario se registró correctamente',
            'user' => $user
        ], 201);
    }


    /**
     * Log the user out (Invalidate the token).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        auth()->logout();

        return response()->json(['message' => 'El usuario cerró sesión correctamente']);
    }

    /**
     * Refresh a token.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function refresh()
    {
        return $this->createNewToken(JWTAuth::refresh(), null, null, null);
    }

    /**
     * Get the authenticated User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function userProfile()
    {
        return response()->json(auth()->user());
    }

    /**
     * Get the token array structure.
     *
     * @param  string $token
     *
     * @return \Illuminate\Http\JsonResponse
     */

    protected function createNewToken($token,$modulo)
    {
        $iPersId = auth()->user()->iPersId;

        $data = DB::table('PERSONAS as p')
        ->leftJoin('TIPO_PERSONAS as tp', 'tp.iTipoPersId', '=', 'p.iTipoPersId')
        ->select('p.cPersNombre', 'p.cPersPaterno', 'p.cPersMaterno', 'p.cPersDocumento', 'p.iTipoIdenId','tp.cTipoPersNombre')
        ->where('p.iPersId', $iPersId)
        ->get();

        return response()->json([
            'access_token' => $token,
            'token_type' => 'bearer',
            'expires_in' => JWTAuth::factory()->getTTL() * 60,
            'user' => auth()->user(),
            'persona' => count($data) > 0 ? $data[0] : null,
            'perfiles' => $this->getPerfiles($modulo),// perfiles = sistemas
            'sistemas' => $this->getSistemas($modulo),
            'roles' => $this->getRoles()

        ]);
    }

    public function getPerfiles($modulo)
    {
        $perfiles = DB::table('CREDENCIALES as c')
            ->join('CREDENCIALES_SISTEMAS as cs', 'cs.iCredId', '=', 'c.iCredId')
            ->join('SISTEMAS as s', 'cs.iSistId', '=', 's.iSistId')
            ->join('PERFILES as p', 'p.iPerfilId', '=', 'cs.iPerfilId')
            ->where([['c.iCredId','=', auth()->user()->iCredId],['s.cSistCodigo','=',$modulo]])
            ->select('p.iPerfilId', 'p.cPerfilCodigo', 'p.cPerfilNombre')
            ->get();

        return $perfiles;
    }

    public function getSistemas($modulo)
    {
        $sistemas = DB::table('CREDENCIALES as c')
            ->join('CREDENCIALES_SISTEMAS as cs', 'cs.iCredId', '=', 'c.iCredId')
            ->join('SISTEMAS as s', 'cs.iSistId', '=', 's.iSistId')
            ->join('PERFILES as p', 'p.iPerfilId', '=', 'cs.iPerfilId')
            ->where([['c.iCredId','=', auth()->user()->iCredId],['s.cSistCodigo','=',$modulo]])
            ->select('p.iPerfilId', 'p.cPerfilCodigo', 'p.cPerfilNombre')
            ->get();

        return $sistemas;
    }

    public function getRoles()
    {
        $roles = DB::table('CREDENCIALES as c')
        ->join('CREDENCIALES_ROLES as cr', 'cr.iCredId', '=', 'c.iCredId')
        ->join('ROLES as r', 'r.iRoleId', '=', 'cr.iRoleId')
        ->where([['c.iCredId','=', auth()->user()->iCredId]])
        ->select('cr.iCredRoleId','r.iRoleId','r.cRoleNombre')
        ->get();

        return $roles;
    }


}
